        ;
        ; Cubos
        ;
        ; by Oscar Toledo G.
	; (c) Copyright Oscar Toledo G. 1991-2024
	; https://nanochess.org/
        ;
        ; Creation date: Early 1991. I was age 12.
        ; Revision date: Feb/12/2024. Ported to MSX/Colecovision.
	;

COLECO: EQU 1   ; Define this to 0 for MSX, 1 for Colecovision

RAM_BASE:	EQU $E000-$7000*COLECO
VDP:		EQU $98+$26*COLECO

PSG:	EQU $FF	; Colecovision

PSG_ADDR:	EQU $A0	; MSX
PSG_DATA:	EQU $A1	; MSX

KEYSEL:	EQU $80
JOYSEL:	EQU $C0
JOY1:	EQU $FC
JOY2:	EQU $FF

    if COLECO
        fname "cubos2_cv.ROM"

	org $8000,$9fff

	dw $aa55	; No BIOS title screen
	dw 0
	dw 0
	dw 0
	dw 0
	dw START

	jp 0		; RST $08
	jp 0		; RST $10
	jp 0		; RST $18
	jp 0		; RST $20
	jp 0		; RST $28
	jp 0		; RST $30
	jp 0		; RST $38

	jp 0		; No NMI handler

    else
        fname "cubos2_msx.ROM"

	org $4000,$5fff

	dw $4241
	dw START
	dw 0
	dw 0
	dw 0
	dw 0
	dw 0
	dw 0

WRTPSG: equ $0093
SNSMAT:	equ $0141

    endif

WRTVDP:
	ld a,b
	out (VDP+1),a
	ld a,c
	or $80
	out (VDP+1),a
	ret

SETWRT:
	ld a,l
	out (VDP+1),a
	ld a,h
	or $40
	out (VDP+1),a
	ret

SETRD:
	ld a,l
	out (VDP+1),a
	ld a,h
        and $3f
	out (VDP+1),a
	ret

WRTVRM:
	push af
	call SETWRT
	pop af
	out (VDP),a
	ret

RDVRM:
        push af
        call SETRD
        pop af
        ex (sp),hl
        ex (sp),hl
        in a,(VDP)
        ret

FILVRM:
	push af
	call SETWRT
.1:	pop af
	out (VDP),a
	push af
	dec bc
	ld a,b
	or c
	jp nz,.1
	pop af
	ret

	; Setup VDP before game
setup_vdp:
	LD BC,$0200
	CALL WRTVDP
        LD BC,$8201     ; No interrupts, disable screen.
	CALL WRTVDP
        LD BC,$0602     ; $1800 for pattern table
	CALL WRTVDP
	LD BC,$FF03	; $2000 for color table
	CALL WRTVDP
	LD BC,$0304	; $0000 for bitmap table
	CALL WRTVDP
	LD BC,$3605	; $1b00 for sprite attribute table
	CALL WRTVDP
	LD BC,$0706	; $3800 for sprites bitmaps
	CALL WRTVDP
        LD BC,$0107     ; Black border
	CALL WRTVDP
        LD HL,$0000
        LD BC,$1800
        XOR A
        CALL FILVRM
        LD HL,$2000
        LD BC,$1800
        LD A,$F1
        CALL FILVRM
        LD HL,$1B00
        LD A,$D0
        CALL WRTVRM
        LD HL,$1800
.1:
        LD A,L
        CALL WRTVRM
        INC HL
        LD A,H
        CP $1B
        JP NZ,.1
        LD BC,$C201     ; No interrupts, enable screen
	CALL WRTVDP
        xor a
        ld (debounce),a
	RET

	;
	; Gets the joystick direction
	; 0 - No movement
	; 1 - Up
	; 2 - Up + right
	; 3 - Right
	; 4 - Right + down
	; 5 - Down
	; 6 - Down + left
	; 7 - Left
	; 8 - Left + Up
	;
GTSTCK:
    if COLECO
        out (JOYSEL),a
	ex (sp),hl
	ex (sp),hl
        in a,(JOY1)
	ld b,a
	in a,(JOY2)
	and b
        and $0f
        ld c,a
        ld b,0
        ld hl,joy_map
        add hl,bc
        ld a,(hl)
        ret

joy_map:
        db 0,0,0,6,0,0,8,7,0,4,0,5,2,3,1,0

    else
	xor a
	call $00d5
	or a
	ret nz
	ld a,1
	call $00d5
	or a
	ret nz
	ld a,2
	jp $00d5
    endif

	; ROM routines I forgot

	; Select address or register in VDP
L0100:
	LD A,L
	OUT (VDP+1),A
	LD A,H
	ADD A,$40
	OUT (VDP+1),A
	RET

L0309:  
        CALL SETWRT
.1:
        LD A,(DE)
        OUT (VDP),A
        INC DE
        DEC BC
        LD A,B
        OR C
        JP NZ,.1
        RET

L0099:      EQU setup_vdp

L0AE8:  ; ???
        RET

L0109:  ; ???
        RET

L0454:      EQU RDVRM
L0447:      EQU WRTVRM

        ;
        ; Draw colored pixel
        ; D = Y-coordinate.
        ; E = X-coordinate.
        ;
L12DA:
        call coor2vdp	; Get coordinate and bitmask.
        ld c,a		; Save bitmask.
        call RDVRM	; Read VRAM.
        or c		; Set pixel.
        call WRTVRM	; Write VRAM.
        set 5,h		; Go to color area.
        ld a,(LEF13)	; Get current color.
        jp WRTVRM	; Set color for 8 pixels.

        ;
        ; Bresenham line drawing
        ; by Oscar Toledo G.
        ; Feel free to use it in your own projects.
        ; Just put my credit somewhere.
        ;
        ; H = Y1 coordinate.
        ; L = X1 coordinate.
        ; D = Y2 coordinate.
        ; E = X2 coordinate.
        ;
L1500:
        ld a,d		
        sub h
        ld b,1
        jr nc,$+6
        ld b,-1
        neg
        exx
        ld b,a		; dy = abs(y2 - y1)
        exx

        ld a,e
        sub l
        ld c,1
        jr nc,$+6
        ld c,-1
        neg
        exx
        ld c,a		; dx = abs(x2 - x1)
        exx

        exx
        ld a,c
        cp b		; dx >= dy?
        jr c,.7		; No, jump.

        ld l,b
        ld h,0
        add hl,hl       ; 2*dy
        ld e,c
        ld d,0
        sbc hl,de       ; -dx
        exx

.1:
        push bc
        push de
        push hl
        ex de,hl
        call L12DA	; Draw a colored pixel on the screen.
        pop hl
        pop de
        pop bc
        ld a,h
        cp d
        jr nz,.2
        ld a,l
        cp e		; Reached endpoint?
        ret z		; Yes, return.
.2:     exx
        bit 7,h
        exx
        jr nz,.5
        ld a,h
        add a,b		; Displace Y-coordinate in Y direction.
        ld h,a
        exx
        ld e,c
        ld d,0
        sla e
        rl d
        sbc hl,de	; d -= dx * 2
        exx
.5:     ld a,l
        add a,c		; Displace X-coordinate in X direction.
        ld l,a
        exx
        ld e,b
        ld d,0
        sla e
        rl d
        add hl,de	; d += dy * 2
        exx
        jr .1

.7:
        ld l,c
        ld h,0
        add hl,hl       ; 2*dx
        ld e,b
        ld d,0
        sbc hl,de       ; -dy
        exx

.3:
        push bc
        push de
        push hl
        ex de,hl
        call L12DA	; Draw a colored pixel on the screen.
        pop hl
        pop de
        pop bc
        ld a,h
        cp d
        jr nz,.4
        ld a,l
        cp e		; Reached endpoint?
        ret z		; Yes, return.
.4:     exx
        bit 7,h		; d < 0?
        exx
        jr nz,.6	; Yes, jump.
        ld a,l
        add a,c		; Displace X-coordinate in X direction.
        ld l,a
        exx
        ld e,b
        ld d,0
        sla e
        rl d
        sbc hl,de	; d -= dy * 2
        exx
.6:     ld a,h
        add a,b		; Displace Y-coordinate in Y direction.
        ld h,a
        exx
        ld e,c
        ld d,0
        sla e
        rl d
        add hl,de	; d += dx * 2
        exx
        jr .3

	;
	; Convert an X,Y coordinate to a bitmap VRAM address.
	; D = Y-coordinate.
	; E = X-coordinate.
	;
	; HL = Final VRAM address.
	; A = Pixel bitmask.
	;
coor2vdp:
        ld a,d
        rrca
        rrca
        rrca
        and $1f
        ld h,a
        ld a,e
        and $f8
        ld l,a
        ld a,d
        and $07
        or l
        ld l,a
        ld a,e
        and $07
        add a,pixel and 255
        ld e,a
        adc a,pixel>>8
        sub e
        ld d,a
        ld a,(de)
        ret

pixel:  db $80,$40,$20,$10,$08,$04,$02,$01

	; Set pixel.
L09B9:
        call coor2vdp
        ld c,a
        call RDVRM
        or c
        call WRTVRM
        RET

	; Erase pixel.
L09E4:  call coor2vdp
        cpl
        ld c,a
        call RDVRM
        and c
        jp WRTVRM

	; Test pixel.
L09FC:  call coor2vdp
        ld c,a
        call RDVRM
        and c
        RET

        ; Read keyboard
L02AA:
.0:
        ld bc,$0800	; Delay.
        dec bc
        ld a,b
        or c
        jr nz,$-3

        xor a		; Read keyboard movement.
        call GTSTCK
        or a
        jr z,.2
        ld a,1		; Read joystick 1 movement.
        call GTSTCK
.2:
        ld b,a
        ld a,(debounce)
        or a		; Is it debouncing?
        jr z,.1		; No, jump.
        dec a		; Yes, countdown.
        ld (debounce),a
        ld b,$ff	; No movement accepted.
.1:
        ld a,$00

        dec b		; 1-Up?
        jr nz,$+4
        ld a,$59	; ASCII Y

        dec b
        dec b		; 3-Right?
        jr nz,$+4
        ld a,$48	; ASCII H

        dec b
        dec b		; 5-Down?
        jr nz,$+4
        ld a,$47	; ASCII G

        dec b
        dec b		; 7-Left?
        jr nz,$+4
        ld a,$54	; ASCII T

        or a		; Any key pressed?
        jr z,.0		; No, jump and wait.
        push af
        ld a,15		; Start debouncing delay.
        ld (debounce),a
        pop af
        RET


START:
        LD SP,STACK

L8000:      	CALL L836A	; Init variables.
L8003:      	CALL L0099	; Init VDP.

	;
	; Draw faces.
	;
	ld hl,face_bitmaps
	ld de,$0000
	call copy_rectangle
	ld hl,face_bitmaps
	ld de,$00e0
	call copy_rectangle
	ld hl,face_colors
	ld de,$2000
	call copy_rectangle
	ld hl,face_colors
	ld de,$20e0
	call copy_rectangle

	call L8984

        ld de,$1080	; Top center of first cube.
        ld b,1  	; One cube at the top.
.1:
        push bc
        push de
        ;
        ; Draw a row of cubes.
        ;
.2:
        push bc
        push de
        call draw_cube
        pop de
        ld a,e
        add a,$18	; Move X-coordinate.
        ld e,a
        pop bc
        djnz .2

        pop de
        ld a,e
        sub $0c		; Half to the left.
        ld e,a
        ld a,d
        add a,$11	; Next row.
        ld d,a
        pop bc
        inc b		; Increase number of cubes.
        ld a,b
        cp 6		; Reached six cubes?
        jp nz,.1	; No, continue drawing.

                LD A,$74
            	LD (LEF13),A
            	LD DE,$A840	; Draw C letter.
            	LD HL,$B840
            	CALL L1500
            	LD DE,$A840
            	LD HL,$A850
            	CALL L1500
            	LD DE,$B840
            	LD HL,$B850
            	CALL L1500
            	LD DE,$B058	; Draw U letter.
            	LD HL,$B858
            	CALL L1500
            	LD DE,$B858
            	LD HL,$B868
            	CALL L1500
            	LD DE,$B068
            	LD DE,$B868
            	CALL L1500
            	LD HL,$B070	; Draw B letter.
            	LD DE,$B07F
            	CALL L1500
            	LD HL,$B470
            	LD DE,$B47F
            	CALL L1500
            	LD HL,$B870
            	LD DE,$B87F
            	CALL L1500
            	LD HL,$B070
            	LD DE,$B870
            	CALL L1500
            	LD HL,$B180
            	LD DE,$B380
            	CALL L1500
            	LD HL,$B580
            	LD DE,$B780
            	CALL L1500
            	LD DE,$B068
            	LD HL,$B868
            	CALL L1500
            	LD HL,$B088	; Draw O letter.
            	LD DE,$B098
            	CALL L1500
            	LD DE,$B098
            	LD HL,$B898
            	CALL L1500
            	LD DE,$B898
            	LD HL,$B888
            	CALL L1500
            	LD HL,$B888
            	LD DE,$B088
            	CALL L1500
            	LD DE,$B0A0	; Draw S letter.
            	LD HL,$B0B0
            	CALL L1500
            	LD DE,$B0A0
            	LD HL,$B4A0
            	CALL L1500
            	LD HL,$B4A0
            	LD DE,$B4B0
            	CALL L1500
            	LD HL,$B4B0
            	LD DE,$B8B0
            	CALL L1500
            	LD HL,$B8A0
            	LD DE,$B8B0
            	CALL L1500

                ; Load the player sprite
                ld hl,$3800
                ld de,L8250
                ld bc,$0020
                call L0309

                ; Load the falling enemy
                ld hl,$3840
                ld de,falling_enemy
                ld bc,$0020
                call L0309

                ; Load the enemy sprite
                ld a,(LFF0D)
                dec a
                ld l,a
                ld h,0
                add hl,hl       ; x2
                add hl,hl       ; x4
                add hl,hl       ; x8
                add hl,hl       ; x16
                add hl,hl
                 ; x32
                ld de,L8270
                add hl,de
                ex de,hl
                ld hl,$3820
                ld bc,$0020
                call L0309

                jp L8290        ; Set sprites.

draw_cube:
        push de
        call draw_left		; 1   1/ \5        
        push de			;     /   \
        call draw_vertical      ; 2 2|\4  /|6
        call draw_right         ; 3  | \ /8|
        pop de			;    3\ I9/7
        call draw_right         ; 4    \I/
        pop de
        call draw_right		; 5
        push de
        call draw_vertical	; 6
        call draw_left		; 7
        pop de
        call draw_left		; 8
        jp draw_vertical	; 9

	;
	; Draw a diagonal line to the left.
	;
draw_left:
        ld b,$06
.1:     call draw_pixel
        dec e
        call draw_pixel
        dec e
        inc d
        djnz .1
        ret

	;
	; Draw a diagonal line to the right.
	;
draw_right:
        ld b,$06
.1:     call draw_pixel
        inc e
        call draw_pixel
        inc e
        inc d
        djnz .1
        ret

	;
	; Draw a vertical line.
	;
draw_vertical:
        ld b,$0b
.1:     call draw_pixel
        inc d
        djnz .1
        ret

draw_pixel:
        push de
        call L09B9
        pop de
        ret

	;
	; Player sprite.
	;
L8250:          db $03,$05,$07,$03  ; $8250
            	db $01,$3F,$EF,$EF  ; $8254
            	db $C7,$C7,$03,$0F  ; $8258
            	db $0C,$0C,$0C,$3C  ; $825C
            	db $80,$40,$C0,$80  ; $8260
            	db $00,$F8,$EE,$EE  ; $8264
            	db $C6,$C6,$80,$E0  ; $8268
            	db $60,$60,$60,$78  ; $826C

        ;
        ; Falling enemy.
        ;
falling_enemy:
	DB $1b,$2d,$2d,$36,$1f,$7d,$9b,$03
	DB $0f,$1f,$3e,$3c,$3c,$3f,$1f,$0f
	DB $00,$00,$00,$00,$00,$80,$80,$82
	DB $02,$06,$06,$0e,$cc,$ec,$fc,$38

	;
	; Enemy 1 sprite.
	;
L8270:
            	db $03,$0F,$1F,$30  ; $8270
            	db $37,$69,$6F,$6C  ; $8274
            	db $6B,$37,$30,$DF  ; $8278
            	db $CF,$63,$60,$90  ; $827C
            	db $C0,$F0,$F8,$0C  ; $8280
            	db $EC,$96,$F6,$36  ; $8284
            	db $D6,$EC,$0C,$FB  ; $8288
            	db $F3,$C6,$06,$09  ; $828C

L85F8:          db $06  
            	db $0F,$07,$05,$07  ; $85F9
            	db $03,$01,$0F,$1B  ; $85FD
            	db $1B,$1B,$03,$06  ; $8601
            	db $06,$06,$0E,$60  ; $8605
            	db $F0,$E0,$A0,$E0  ; $8609
            	db $C0,$80,$F0,$D8  ; $860D
            	db $D8,$D8,$C0,$60  ; $8611
            	db $60,$60,$70  ; $8615
L8620:
                db $0F,$3F  
            	db $09,$39,$0F,$3F  ; $8622
            	db $0B,$3C,$0F,$3F  ; $8626
            	db $0F,$3F,$0F,$3F  ; $862A
            	db $0F,$3F,$F0,$FC  ; $862E
            	db $90,$9C,$F0,$FC  ; $8632
            	db $D0,$3C,$F0,$FC  ; $8636
            	db $F0,$FC,$F0,$FC  ; $863A
            	db $F0,$FC  ; $863E

L8648:
                db $01,$03  
            	db $05,$07,$0F,$01  ; $864A
            	db $3F,$77,$77,$73  ; $864E
            	db $73,$F3,$0F,$06  ; $8652
            	db $06,$0E,$80,$C0  ; $8656
            	db $A0,$E0,$F0,$80  ; $865A
            	db $FC,$EE,$EE,$CE  ; $865E
            	db $CE,$CF,$F0,$60  ; $8662
            	db $60,$70  ; $8666

L8670:
                db $81,$41  
            	db $21,$11,$09,$05  ; $8672
            	db $03,$FE,$03,$05  ; $8676
            	db $09,$11,$21,$41  ; $867A
            	db $81,$00,$02,$04  ; $867E
            	db $08,$10,$20,$40  ; $8682
            	db $80,$FF,$80,$40  ; $8686
            	db $20,$10,$08,$04  ; $868A
            	db $02,$00  ; $868E

L8290:      	LD HL,$41C2	; Sprites 16x16 pixels.
            	CALL L0100
            	LD HL,$1B00
            	LD A,$08	; Y-coordinate for player.
            	CALL L8918
            	LD A,$78	; X-coordinate for player.
            	CALL L8918
            	LD A,$00	; Sprite frame for the player.
            	CALL L8918
            	LD A,$06	; Color for the player.
            	CALL L8918
            	LD A,$4C	; Y-coordinate for enemy 1.
            	CALL L8918
            	LD A,$00
            	LD (LFF10),A
            	LD A,$48	; X-coordinate for enemy 1.
            	CALL L8918
            	LD A,$04	; Sprite frame for enemy.
            	CALL L8918
                CALL L8C1B      ; Generate random color.
            	LD A,$4C	; Y-coordinate for enemy 2.
            	CALL L8918
            	LD A,$A8	; X-coordinate for enemy 2.
            	CALL L8918
            	LD A,$04	; Sprite frame for enemy.
            	CALL L8918
                CALL L8C1B      ; Generate random color.
            	LD A,$D1
            	CALL L8918
L82E0:
	call L02AA	; Read the keyboard.
	cp $54
	jr nz,$+5
	ld de,$eff4	; y = -17, x = -12
	cp $59
	jr nz,$+5
	ld de,$ef0c	; y = -17, x = +12
	cp $47
	jr nz,$+5
	ld de,$11f4	; y = +17, x = -12
	cp $48
	jr nz,$+5
	ld de,$110c	; y = +17, x = +12

	ld hl,$1b00
	call L0454	; Read Y-coordinate of the player.
	add a,d		; Add Y offset.
	ld d,a
	inc hl
	call L0454	; Read X-coordinate of the player.
	add a,e		; Add X offset.
	ld e,a

	ld hl,valid_positions
	ld b,15
.1:
	ld a,(hl)
	cp d
	inc hl
	jr nz,.2
	ld a,(hl)
	cp e
	jr z,.3
.2:	
	inc hl
	djnz .1
	jp L82F7	; Invalid movement.

	; Valid movement.
.3:
	ld hl,$1b00
	ld a,d
	call L0447	; Update Y-coordinate of the player.
	inc hl
	ld a,e
	call L0447	; Update X-coordinate of the player.
	ld a,d
	add a,$0e
	ld d,a
	ld a,e
	add a,$06
	ld e,a
	push de
	call L8345	; Check if it filled another pyramid square.
	pop de
	call L86A8	; Draw footprint.
L82F7:
      	CALL L832A
        jp L843E

valid_positions:
	db $08,$78
	db $19,$6c,$19,$84
	db $2a,$60,$2a,$78,$2a,$90
	db $3b,$54,$3b,$6c,$3b,$84,$3b,$9c
	db $4c,$48,$4c,$60,$4c,$78,$4c,$90,$4c,$a8

L832A:      	CALL L83A4
            	LD E,A
            	NOP
            	NOP
            	JP L88E0

	;
	; Generate a pseudorandom number.
	;
L8333:      	LD A,R
            	CP C
            	RET C
            	SRA A
            	CP C
            	RET C
            	PUSH BC
            	LD B,A
L833D:      	DJNZ L833D
            	POP BC
            	JR L8333

L8345:      	PUSH DE
            	CALL L09FC
            	POP DE
            	RET NZ
                LD HL,LFF0F	; Increase number of filled squares.
            	INC (HL)
            	RET

L836A:      	XOR A
                LD (LFF0F),A    ; Reset number of squares filled.
                LD (LFF0E),A
            	INC A
            	LD (LFF0D),A
                CALL L0099
            	JP L0AE8

L83A4:      	LD A,(LFF0F)	; Get number of filled squares?
            	CP $0F		; Is it 15?
            	JP Z,L85C1	; Yes, jump to advance level.
            	JP L83EB	; No, continue.

L83AF:      	CALL L0454	; Read byte from VRAM.
            	LD B,A
            	LD HL,$1B04	; Y-coordinate of the enemy 1.
            	CALL L0454	; Read byte from VRAM.
            	CP B		; enemy1_y < player_y
            	JP C,L83C5	; Yes, jump.
            	SUB $11		; No, goes upwards toward the player.
L83BF:      	CALL L0447	; Update byte in VRAM.
            	JP L83CA

L83C5:      	ADD A,$11	; Goes downwards toward the player.
            	JP L83BF

L83CA:      	LD HL,$1B01	; X-coordinate of the player.
            	CALL L0454	; Read byte from VRAM.
            	LD B,A
            	LD HL,$1B05	; X-coordinate of the enemy 1.
            	CALL L0454	; Read byte from VRAM.
            	CP B		; enemy1_x < player_x
            	JP C,L83E3	; Yes, jump.
            	SUB $0C		; No, goes left toward the player.
L83DD:      	CALL L0447	; Update byte in VRAM.
            	JP L83E8

L83E3:      	ADD A,$0C	; Goes right toward the player.
            	JP L83DD

L83E8:      	JP L8402	; Jump to move 2nd enemy.

L83EB:      	LD A,(LFF0E)
            	CPL		; Switch turn of enemy.
            	LD (LFF0E),A
            	CP $FF		; Can move enemies?
            	RET Z		; No, return.
            	LD HL,$1B00	; Point to Y-coordinate of the player.
            	JP L83AF

L8402:      	LD HL,$1B00	; Y-coordinate of the player.
            	CALL L0454	; Read byte from VRAM.
            	LD B,A
            	LD HL,$1B08	; Y-coordinate of 2nd enemy.
            	CALL L0454	; Read byte from VRAM.
            	CP B		; enemy2_y < player_y
            	JP C,L841B	; Yes, jump.
            	SUB $11		; It goes upwards toward the player.
L8415:      	CALL L0447	; Update byte in VRAM.
            	JP L8420

L841B:      	ADD A,$11	; It goes downwards.
            	JP L8415	; Update byte in VRAM.

L8420:      	LD HL,$1B01	; X-coordinate of the player.
            	CALL L0454	; Read byte from VRAM.
            	LD B,A
            	LD HL,$1B09	; X-coordinate of 2nd enemy.
            	CALL L0454	; Read byte from VRAM.
            	CP B		; enemy2_x < player_x
            	JP C,L8439	; Yes, jump.
            	SUB $0C		; It goes left toward the player.
L8433:      	JP L0447

L8439:      	ADD A,$0C	; It goes right toward the player.
            	JP L8433	; Update and return to calling code.

L843E:      	LD HL,$1B00	; Pointer to Y-coordinate of the player.
            	CALL L0454
            	LD B,A
            	LD HL,$1B04	; Pointer to Y-coordinate of the enemy 1.
            	CALL L0454
            	CP B		; Is it the same?
            	CALL Z,L845C	; Yes, call subroutine.
            	LD HL,$1B08	; Pointer to Y-coordinate of the enemy 2.
            	CALL L0454
            	CP B		; Is it the same?
            	CALL Z,L846E	; Yes, call subroutine.
            	JP L82E0	; Jump back to player movement.

L845C:      	LD HL,$1B01	; Pointer to X-coordinate of the player.
            	CALL L0454	; Read byte from VRAM.
            	LD C,A
            	LD HL,$1B05	; Pointer to X-coordinate of the enemy 1.
            	CALL L0454	; Read byte from VRAM.
            	CP C		; Are the same?
            	JP Z,L8480	; Yes, player defeated.
            	RET

L846E:      	LD HL,$1B01	; Pointer to X-coordinate of the player.
            	CALL L0454	; Read byte from VRAM.
            	LD C,A
            	LD HL,$1B09	; Pointer to X-coordinate of the enemy 2.
            	CALL L0454	; Read byte from VRAM.
            	CP C		; Are the same?
            	JP Z,L8480	; Yes, player defeated.
            	RET

	;
	; Player defeated.
	;
L8480:      	LD A,$F6
            	LD (LEF13),A
            	LD HL,$8020	; Letter P.
            	LD DE,$9020
            	CALL L1500
            	LD HL,$8020
            	LD DE,$8030
            	CALL L1500
            	LD HL,$8820
            	LD DE,$8830
            	CALL L1500
            	LD HL,$8030
            	LD DE,$8830
            	CALL L1500
            	LD HL,$8438	; Letter I.
            	LD DE,$8638
            	CALL L1500
            	LD HL,$8838
            	LD DE,$9038
            	CALL L1500
            	LD HL,$8840	; Letter E.
            	LD DE,$8850
            	CALL L1500
            	LD HL,$8840
            	LD DE,$9040
            	CALL L1500
            	LD HL,$9040
            	LD DE,$9050
            	CALL L1500
            	LD HL,$8C40
            	LD DE,$8C50
            	CALL L1500
            	LD HL,$8858	; Letter R.
            	LD DE,$9058
            	CALL L1500
            	LD HL,$8858
            	LD DE,$8868
            	CALL L1500
            	LD HL,$8868
            	LD DE,$8C68
            	CALL L1500
            	LD HL,$8C58
            	LD DE,$8C68
            	CALL L1500
            	LD HL,$8C58
            	LD DE,$9068
            	CALL L1500
            	LD DE,$8870	; Letter D.
            	LD HL,$887C
            	CALL L1500
            	LD DE,$9070
            	LD HL,$907C
            	CALL L1500
            	LD DE,$8870
            	LD HL,$9070
            	CALL L1500
            	LD HL,$8A80	
            	LD DE,$8E80
            	CALL L1500
            	LD HL,$887C
            	LD DE,$8A80
            	CALL L1500
            	LD DE,$8E80
            	LD HL,$907C
            	CALL L1500
            	LD DE,$8888	; Letter E.
            	LD HL,$9088
            	CALL L1500
            	LD DE,$8888
            	LD HL,$8898
            	CALL L1500
            	LD DE,$8C88
            	LD HL,$8C98
            	CALL L1500
            	LD DE,$9088
            	LD HL,$9098
            	CALL L1500
            	LD HL,$88A0	; Letter S.
            	LD DE,$88B0
            	CALL L1500
            	LD HL,$8CA0
            	LD DE,$8CB0
            	CALL L1500
            	LD HL,$90A0
            	LD DE,$90B0
            	CALL L1500
            	LD HL,$88A0
            	LD DE,$8CA0
            	CALL L1500
            	LD HL,$8CB0
            	LD DE,$90B0
            	CALL L1500
            	LD HL,$84C0	; Exclamation mark.
            	LD DE,$8BC0
            	CALL L1500
            	LD HL,$8DC0
            	LD DE,$90C0
            	CALL L1500
            	LD B,$05
L85A7:      	PUSH BC
            	LD BC,$0000
L85AB:      	DEC BC
            	LD A,B
            	OR C
            	JR NZ,L85AB
            	POP BC
            	DJNZ L85A7
            	CALL L0109
            	JP L8000

	;
	; Level completed.
	;
L85C1:          LD HL,LFF0D
            	INC (HL)	; Increment current level.
            	LD A,(HL)
            	CP $06		; Level 6?
                JR NZ,$+4
                LD (HL),$01     ; Go back to level 1.
            	XOR A
            	LD (LFF0F),A
            	LD (LFF0E),A
                LD A,$FD        ; White lines with magenta background.
            	LD (LEF13),A
            	LD HL,$8038	; Letter G.
            	LD DE,$9038
            	CALL L1500
            	LD HL,$8038
            	LD DE,$8048
            	CALL L1500
            	LD HL,$8048
            	LD DE,$8348
            	CALL L1500
            	LD HL,$9038
            	LD DE,$9048
            	CALL L1500
            	LD DE,$8948
            	LD HL,$9048
            	CALL L1500
            	LD HL,$8940
            	LD DE,$8948
            	CALL L1500
            	LD HL,$8850	; Letter A.
            	LD DE,$8860
            	CALL L1500
            	LD HL,$8860
            	LD DE,$9060
            	CALL L1500
            	LD HL,$8850
            	LD DE,$9050
            	CALL L1500
            	LD HL,$8C50
            	LD DE,$8C60
            	CALL L1500
            	LD HL,$8868	; Letter N.
            	LD DE,$9068
            	CALL L1500
            	LD HL,$8878
            	LD DE,$9078
            	CALL L1500
            	LD HL,$8868
            	LD DE,$9078
            	CALL L1500
            	LD HL,$8880	; Letter A.
            	LD DE,$8890
            	CALL L1500
            	LD HL,$8880
            	LD DE,$9080
            	CALL L1500
            	LD HL,$8890
            	LD DE,$9090
            	CALL L1500
            	LD HL,$8C80
            	LD DE,$8C90
            	CALL L1500
            	LD HL,$8898	; Letter S.
            	LD DE,$88A8
            	CALL L1500
            	LD HL,$8898
            	LD DE,$8C98
            	CALL L1500
            	LD HL,$8C98
            	LD DE,$8CA8
            	CALL L1500
            	LD HL,$8CA8
            	LD DE,$90A8
            	CALL L1500
            	LD HL,$9098
            	LD DE,$90A8
            	CALL L1500
            	LD HL,$84B8	; Exclamation sign.
            	LD DE,$8CB8
            	CALL L1500
            	LD HL,$8EB8
            	LD DE,$90B8
            	CALL L1500
            	LD B,$05	; Five big delays.
L8B88:      	PUSH BC
            	LD BC,$0000
L8B8C:      	DEC BC
            	LD A,B
            	OR C
            	JR NZ,L8B8C
            	POP BC
            	DJNZ L8B88
            	JP L8003

	;
	; Draw a footprint on a square.
	;
L86A8:      	CALL L869C
		INC E
            	CALL L869C
            	INC E
            	CALL L869C
            	INC E
            	CALL L869C
            	INC E
            	CALL L869C
            	INC D
            	CALL L869C
            	DEC E
            	CALL L869C
            	DEC E
            	CALL L869C
            	DEC E
            	CALL L869C
            	INC D
            	DEC E
            	CALL L869C
            	INC E
            	CALL L869C
            	INC E
            	CALL L869C
            	INC E
	;
	; Draw a single pixel for the footprint.
	;
L869C:      	LD A,$F1
            	LD (LEF13),A
            	PUSH DE
            	CALL L12DA
            	POP DE
            	RET

	;
	; Bitmaps and color for the face.
	;
face_bitmaps:
 db $ff,$ff,$fc,$fe,$ff,$f9,$fd,$fe
 db $f1,$a4,$a9,$52,$54,$55,$3f,$7f
 db $2d,$09,$22,$00,$92,$20,$ff,$fc
 db $ff,$3f,$2f,$8f,$1f,$4f,$1f,$2f
 db $f9,$fc,$fe,$f9,$fa,$f8,$fa,$fa
 db $3f,$ff,$3f,$bf,$c7,$81,$1f,$c1
 db $ff,$fc,$ff,$ff,$e3,$81,$fc,$83
 db $8f,$0f,$af,$1f,$5f,$1f,$5f,$5f
 db $fa,$fa,$fc,$fe,$fe,$fe,$fe,$ff
 db $b3,$ff,$fe,$fe,$fd,$fd,$fe,$7f
 db $e5,$ff,$ff,$ff,$ff,$7f,$ff,$fe
 db $5f,$5f,$3f,$7f,$7f,$7f,$7f,$ff
 db $ff,$ff,$ff,$ff,$ff,$ff,$ff,$ff
 db $7c,$7c,$bf,$df,$ef,$f7,$f8,$f8
 db $3e,$3e,$fc,$f8,$f0,$e0,$ff,$ff
 db $ff,$ff,$ff,$ff,$ff,$ff,$ff,$ff
face_colors:
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $a1,$a1,$a1,$a1,$a1,$a1,$b1,$b1
 db $a1,$a1,$a1,$f1,$a1,$a1,$91,$91
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $b1,$b1,$b1,$b1,$b1,$b1,$b1,$b1
 db $91,$91,$91,$91,$91,$91,$91,$91
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $b1,$b1,$b1,$b1,$b1,$b1,$b1,$b1
 db $91,$91,$91,$91,$91,$91,$91,$91
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1
 db $b1,$b1,$b1,$b1,$b1,$b1,$b1,$b9
 db $91,$91,$98,$98,$98,$98,$81,$81
 db $a1,$a1,$a1,$a1,$a1,$a1,$a1,$a1

copy_rectangle:
	ld b,4
.1:	push bc
	push de
	ex de,hl
	ld b,32
.2:	ld a,(de)
	call L0447
	inc hl
	inc de
	djnz .2
	ex de,hl
	pop de
	inc d
	pop bc
	djnz .1
	ret

	;
	; Create the third enemy.
	;
L88E0:      	LD A,(LFF10)
            	CP $FE
            	JP Z,L8926
                LD C,$7F        ; Random range.
                CALL L8333      ; Generate a random number.
                CP $40          ; Is it less than $40?
                RET C           ; Yes, return without moving.
            	LD HL,$1B0C	; Pointer to 4th sprite on Sprite Attribute Table.
            	LD A,$08	; Y-coordinate.
            	CALL L0447
            	INC HL
            	LD A,$78	; X-coordinate.
            	CALL L0447
            	INC HL
                LD A,$08        ; Sprite frame.
            	CALL L0447
            	INC HL
            	LD A,$0E	; Color.
                CALL L0447
            	LD A,$FE	; Mark enemy as initialized.
            	LD (LFF10),A
            	RET

L8918:      	CALL L0447	; Write byte to VRAM.
            	INC HL		; Increment pointer.
            	RET

	;
	; Move the third enemy.
	;
L8926:      	LD HL,$1B0C	; Y-coordinate.
            	CALL L0454
            	ADD A,$11	; Move downward.
            	CALL L8971
            	LD C,$7F	; Generate random number.
            	CALL L8333
            	CP $3F		; Less than $3f?
            	JR C,L8945
            	LD HL,$1B0D
            	CALL L0454
            	SUB $0C		; Move to the left.
            	JP L894D

L8945:      	LD HL,$1B0D
            	CALL L0454
            	ADD A,$0C	; Move to the right.
L894D:      	CALL L0447	; Update in VRAM.
            	LD HL,$1B00	; Y-coordinate of the player.
            	CALL L0454
            	LD B,A
            	LD HL,$1B0C	; Y-coordinate of the third enemy.
            	CALL L0454
            	CP B		; Same?
            	RET NZ		; No, return.
            	LD HL,$1B01	; X-coordinate of the player.
            	CALL L0454
            	LD B,A
            	LD HL,$1B0D	; X-coordinate of the third enemy.
            	CALL L0454
            	CP B		; Same?
            	RET NZ		; No, return.
            	JP L8480	; Player defeated.

L8971:      	CALL L0447	; Write to VRAM.
            	CP $5D		; Went out of the pyramid?
            	RET NZ		; No, return.
            	LD A,$D1	; Make it disappear from screen.
            	CALL L0447
            	LD A,$00	; Third enemy now can appear again.
            	LD (LFF10),A
            	POP IX		; Pop return address.
            	RET		; Return to previous caller.

	;
	; Draw the visual instructions for keyboard usage.
	;
L8984:      	LD A,$F1
            	LD (LEF13),A
            	LD HL,$4010	; Draw a square (top left)
            	LD DE,$4020
            	CALL L1500
            	LD HL,$4010
            	LD DE,$5010
            	CALL L1500
            	LD HL,$5010
            	LD DE,$5020
            	CALL L1500
            	LD HL,$4020
            	LD DE,$5020
            	CALL L1500
            	LD HL,$5810	; Draw a square (bottom left)
            	LD DE,$5820
            	CALL L1500
            	LD HL,$5810
            	LD DE,$6810
            	CALL L1500
            	LD HL,$6810
            	LD DE,$6820
            	CALL L1500
            	LD HL,$6820
            	LD DE,$5820
            	CALL L1500
            	LD HL,$4028	; Draw a square (top right)
            	LD DE,$4038
            	CALL L1500
            	LD HL,$4038
            	LD DE,$5038
            	CALL L1500
            	LD HL,$5028
            	LD DE,$5038
            	CALL L1500
            	LD HL,$4028
            	LD DE,$5028
            	CALL L1500
            	LD HL,$5828	; Draw a square (bottom right)
            	LD DE,$5838
            	CALL L1500
            	LD HL,$5838	
            	LD DE,$6838
            	CALL L1500
            	LD HL,$6838
            	LD DE,$6828
            	CALL L1500
            	LD HL,$5828
            	LD DE,$6828
            	CALL L1500
            	LD HL,$5020	; Connect top-left with bottom-right
            	LD DE,$5828
            	CALL L1500
            	LD HL,$5028	; Connect top-right with bottom-left
            	LD DE,$5820
            	CALL L1500
            	LD HL,$4414	; Draw a T.
            	LD DE,$441C
            	CALL L1500
            	LD HL,$4418
            	LD DE,$4C18
            	CALL L1500
            	LD HL,$442C	; Draw a Y.
            	LD DE,$4830
            	CALL L1500
            	LD HL,$4434
            	LD DE,$4830
            	CALL L1500
            	LD HL,$4830
            	LD DE,$4C30
            	CALL L1500
            	LD HL,$5C2C	; Draw a H.
            	LD DE,$642C
            	CALL L1500
            	LD HL,$5C34
            	LD DE,$6434
            	CALL L1500
            	LD HL,$602C
            	LD DE,$6034
            	CALL L1500
            	LD HL,$5C14	; Draw a G.
            	LD DE,$5C1C
            	CALL L1500
            	LD HL,$5C14
            	LD DE,$6414
            	CALL L1500
            	LD HL,$6414
            	LD DE,$641C
            	CALL L1500
            	LD HL,$641C
            	LD DE,$611C
            	CALL L1500
            	LD HL,$6118
            	LD DE,$611C
            	CALL L1500
            	LD HL,$5C1C
            	LD DE,$5D1C
	      	CALL L1500
            	LD HL,$07D0
                LD DE,L8270
            	CALL L8BA6	; Draw sprite as bitmap.
	      	LD HL,$0AD0	; Draw 2nd enemy.
                LD DE,L85F8
            	CALL L8BA6
            	LD HL,$0DD0	; Draw 3rd enemy.
                LD DE,L8620
            	CALL L8BA6
            	LD HL,$10D0	; Draw 4th enemy.
                LD DE,L8648
            	CALL L8BA6
            	LD HL,$13D0	; Draw 5th enemy.
                LD DE,L8670
            	JP L8BA6

L8BA6:      	LD B,$08	; 8 pixel rows for top-left character.
L8BA8:      	LD A,(DE)
            	CALL L8C06
            	INC HL
            	INC DE
            	DJNZ L8BA8
            	LD A,L
            	LD B,$08
            	SUB B
            	LD L,A
            	LD A,H
            	INC A		; Go to the next screen row.
            	LD H,A
            	LD B,$08	; 8 pixel rows for bottom-left character.
L8BBA:      	LD A,(DE)
            	CALL L8C06
            	INC HL
            	INC DE
            	DJNZ L8BBA
            	LD A,H
            	DEC A		; Go back to the previous row...
            	LD H,A		; ...but already advanced one column.
            	LD B,$08	; 8 pixel rows for top-right character.
L8BC7:      	LD A,(DE)
            	CALL L8C06
            	INC HL
            	INC DE
            	DJNZ L8BC7
            	LD A,L
            	LD B,$08
            	SUB B
            	LD L,A
            	LD A,H
            	INC A		; Go to the next screen row.
            	LD H,A
            	LD B,$08	; 8 pixel rows for bottom-right character.
L8BD9:      	LD A,(DE)
            	CALL L8C06
            	INC HL
            	INC DE
            	DJNZ L8BD9
            	RET

L8C06:      	CALL L0447	; Write bitmap byte to VRAM.
            	PUSH HL
            	PUSH DE
            	LD DE,$2000	; Switch to color plane.
            	ADD HL,DE
            	LD A,$C1	; Green color.
            	AND $F0		; Nothing like doing operations...
            	OR $01		; ...to get the same value.
            	CALL L0447	; Write color byte to VRAM.
            	POP DE
            	POP HL
            	RET

L8C1B:      	LD A,R		; Read the R register.
            	AND $0F		; Extract only the lower nibble.
                CP $02          ; Not visible color?
                JP C,L8C1B      ; Yes, try another.
            	JP L8918	; Write to VRAM.

        ORG RAM_BASE

LEF13:      RB 1	; Color for drawing pixels.
LFF0D:      RB 1	; Level number.
LFF0E:      RB 1	; Turn for enemies.
LFF0F:      RB 1	; Number of filled squares.
LFF10:      RB 1	; Status of the third enemy.
			; ($00- Not created, $fe- Created)

debounce:       rb 1

STACK:  EQU RAM_BASE+1024

